---
myst:
  html_meta:
    keywords: LaTeX, programmation, commandes, macros, code source, code source des commandes
---
# Comment trouver la définition des commandes LaTeX ?

Il y a plusieurs raisons de vouloir connaître les définitions des commandes LaTeX : de la simple "curiosité" au besoin pressant de corriger quelque chose pour qu'il "fonctionne comme vous le voulez". 

## Avec les commandes de base

En utilisant un exécutable TeX quelconque en mode interactif dans un terminal, la réponse la plus simple est d'essayer la commande `\show`. En voici un exemple (dont la sortie est réorganisée avec des passages à la ligne et indentations par rapport à ce que donne réellement TeX) :

```text
*\makeatletter
*\show\protected@edef
> \protected@edef=macro:
->\let \@@protect \protect
  \let \protect \@unexpandable@protect
  \afterassignment \restore@protect \edef .
```

Mais qu'en est-il de `\@unexpandable@protect` apparaissant dans ce résultat ? Continuons ici l'analyse :

```text
*\show\@unexpandable@protect
> \@unexpandable@protect=macro:
->\noexpand \protect \noexpand .
```

Nous commençons ici à voir comment fonctionne une partie du mécanisme de protection (on peut probablement deviner sans trop de risque ce que fait `\restore@protect`).

De nombreuses commandes du noyau sont déclarées robustes :

```text
*\show\texttt
> \texttt=macro:
->\protect \texttt  .
```

Dès lors, `\show` n'est pas d'une grande aide. Définissez une commande `\pshow` comme indiqué ci-dessous, et exécutez simplement la commande pour trouver sa définition :

```text
*\def\pshow#1{{\let\protect\show #1}}
*\pshow\texttt
> \texttt =\long macro:
#1->\ifmmode \nfss@text {\ttfamily #1}%
    \else \hmode@bgroup \text@command {#1}%
          \ttfamily \check@icl #1\check@icr
    \expandafter \egroup \fi .
```

Notez que le nom de la commande protégée est celui de la commande de "base", avec un espace. Ceci est visible de manière cryptique, à quelques endroits ci-dessus (encore une fois, la sortie a été réorganisée).

## Avec le programme `latexdef`

Le programme `latexdef` (ou `texdef`, la même commande avec un nom différent) fera tout cela pour vous et renverra les résultats de manière légèrement plus ordonnée que LaTeX lui-même. L'exemple (avec notre légère réorganisation du format de la sortie) illustre que `latexdef` pense bien à traiter la protection de la commande.

```text
$ latexdef texttt

\texttt:
macro:->\protect \texttt

\texttt:
#1->\ifmmode \nfss@text {\ttfamily #1}%
    \else \hmode@bgroup \text@command {#1}%
          \ttfamily \check@icl #1\check@icr
    \expandafter \egroup \fi .
```

Avec l'option `-s`, `latexdef` vous donnera un emplacement de source sans donner le détail du codage réel.

```text
$ latexdef -s texttt
% latex.ltx, line 11060:
\DeclareTextFontCommand{\texttt}{\ttfamily}
```

Les environnements transmettent également leurs détails à `latexdef` (avec toujours notre réorganisation de la sortie)

```text
$ latexdef -E itemize

\itemize:
macro:->\ifnum \@itemdepth >\thr@@ \@toodeep
  \else \advance \@itemdepth \@ne
    \edef \@itemitem {labelitem\romannumeral \the \@itemdepth}%
    \expandafter \list \csname \@itemitem \endcsname
      {\def \makelabel ##1{\hss \llap {##1}}}%
  \fi

\enditemize:
macro:->\global \advance \@listdepth \m@ne \endtrivlist
```

## Avec les fichiers `ltx` et `dtx`

Avec un bon éditeur de texte, la même enquête peut être menée en examinant le fichier `latex.ltx` (qui se trouve généralement dans le répertoire `tex/latex/base` pour un système utilisant la [TDS](/5_fichiers/emplacements/la_tds)).

En fait, `latex.ltx` est le produit d'un processus `docstrip` sur un grand nombre de [fichiers dtx](/1_generalites/documentation/documents/documents_extensions/fichiers_sources_dtx.md) ce que vous pouvez également étudier. Une distribution de LaTeX inclut un fichier `source2e.tex` et la plupart des systèmes le conservent, toujours dans `tex/latex/base`. Le fichier `source2e.tex` peut être traité pour fournir une liste complète des sources du noyau LaTeX (en fait, le processus n'est pas tout à fait évident mais le fichier produit des messages vous indiquant ce qu'il faut faire). Le résultat est un énorme document, avec un index par numéro de ligne des séquences de contrôle du noyau entier et un index séparé des changements enregistrés dans chacun des fichiers depuis que l'équipe LaTeX a pris le relais.

Disposer du noyau imprimé pourrait une bonne chose, surtout si on ignore la question du volume de ce document. Cependant, les commentaires du code sont très inégaux : ils vont de "bons" à "inexistants" (sans compter ceux issus de conversion automatique de documentations plus anciennes). Dans les faits, chaque fichier `dtx` du module du noyau sera traité séparément par LaTeX, de sorte que vous n'aurez pas à travailler avec l'ensemble de `source2e`. Vous pouvez facilement déterminer quel module définit la commande qui vous intéresse : utilisez votre éditeur de texte pour trouver la définition dans `latex.ltx`. Ensuite recherchez à partir de ce point une ligne qui commence par :

```text
%%% From File:  
```

Cette ligne vous indique quel fichier `dtx` contient la définition qui vous intéresse. En procédant ainsi pour `\protected@edef`, nous trouvons :

```text
%%% From File: ltdefns.dtx
```

Lorsque nous l'examinons, `ltdefns.dtx` s'avère contenir une discussion sur les méthodes de gestion de la protection avec `\protect`. Ce fichier contient également de la documentation LaTeX 2.09 automatiquement convertie.

Bien entendu, le noyau n'est pas tout LaTeX : votre commande peut être définie dans l'un des fichiers de classe ou d'extension de LaTeX. Par exemple, nous trouvons une définition de `\thebibliography` dans la classe <ctanpkg:article>, mais il n'y a pas de ``article.dtx``. Certains de ces fichiers sont générés à partir de parties du noyau, d'autres à partir d'autres fichiers de la distribution. Cela se retrouve en regardant le début du fichier. Pour `article.cls`, on trouve :

```text
%% This is file `article.cls',
%% generated with the docstrip utility.
%%
%% The original source files were:
%%
%% classes.dtx  (with options: `article')
```

Nous devons donc compiler `classes.dtx` pour voir la définition dans son contexte. 

Il faut noter que tous ces fichiers `.dtx` se trouvent sur le [CTAN](/1_generalites/documentation/le_catalogue_du_ctan.md) et font partie de la distribution principale de LaTeX.

:::{sources}
[The definitions of LaTeX commands](faquk:FAQ-ltxcmds)
:::
